/**
 * \file caam_op_wrap_unwrap.c
 *
 * \brief Architecture specific implementation of functions relevant for wrap and unwrap
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include "caam.h"
#include <linux/caam_ee.h>
#include <sdc/arch/errors.h>

#ifdef CAAM_EE_FLAG_INTERLEAVED_OP_BIT
#ifdef CAAM_EE_FLAG_NO_CLEAR_BIT
    #define CAAM_EE_WRAP_DEFAULT_FLAGS (CAAM_EE_FLAG_NO_CLEAR_BIT | CAAM_EE_FLAG_INTERLEAVED_OP_BIT)
#else
    #define CAAM_EE_WRAP_DEFAULT_FLAGS CAAM_EE_FLAG_INTERLEAVED_OP_BIT
#endif
#else
#ifdef CAAM_EE_FLAG_NO_CLEAR_BIT
    #define CAAM_EE_WRAP_DEFAULT_FLAGS CAAM_EE_FLAG_NO_CLEAR_BIT
#else
    #define CAAM_EE_WRAP_DEFAULT_FLAGS 0
#endif
#endif

#define CAAM_EE_WRAP_INIT_UPDATE_FINALIZE_FLAG 0  /* No interleaving mode is used */

/* define default type for wrap/unwrap in IMX6 */
static const sdc_wrap_unwrap_type_t sdc_wrap_unwrap_default_type =
{SDC_WRAP_ALG_AES, SDC_WRAP_BLK_CCM, CAAM_EE_CIPHER_AES_FLAG, CAAM_EE_CIPHER_AES_IDX, CAAM_EE_WRAP_BLOCK_CCM_FLAG, CAAM_EE_WRAP_BLOCK_CCM_IDX};


/* defined in sdc_arch.h */
const sdc_wrap_unwrap_type_t *sdc_arch_wrap_unwrap_get_default(void)
{
    return &sdc_wrap_unwrap_default_type;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_type_alloc(sdc_wrap_unwrap_type_t **type)
{
    *type = malloc(sizeof(sdc_wrap_unwrap_type_t));
    if (!(*type))
        return SDC_NO_MEM;

    /* initialize with default */
    memcpy(*type, &sdc_wrap_unwrap_default_type, sizeof(sdc_wrap_unwrap_type_t));

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_type_free(sdc_wrap_unwrap_type_t *type)
{
    free (type);

    return SDC_OK;
}

static sdc_error_t caam_map_sdc_wrap_unwrap_alg (sdc_wrap_unwrap_alg_t alg, uint32_t *flag, uint32_t *idx, sdc_wrap_unwrap_blk_t *default_blk)
{
    sdc_error_t err = SDC_OK;

    switch(alg) {
    case SDC_WRAP_ALG_AES:
        *flag = CAAM_EE_CIPHER_AES_FLAG;
        *idx =  CAAM_EE_CIPHER_AES_IDX;
        *default_blk = SDC_WRAP_BLK_CCM;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}

static sdc_error_t caam_map_sdc_wrap_unwrap_blk (sdc_wrap_unwrap_blk_t blk, uint32_t *flag, uint32_t *idx)
{
    sdc_error_t err = SDC_OK;

    switch(blk) {
    case SDC_WRAP_BLK_CCM:
        *flag = CAAM_EE_WRAP_BLOCK_CCM_FLAG;
        *idx =  CAAM_EE_WRAP_BLOCK_CCM_IDX;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_type_set_algorithm(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t alg)
{
    sdc_error_t err = SDC_OK;
    sdc_wrap_unwrap_type_t tmp;

    tmp.alg = alg;
    err = caam_map_sdc_wrap_unwrap_alg(tmp.alg, &tmp.arch_alg_flag,
                                       &tmp.arch_alg_idx, &tmp.blk);

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_type_set_block_mode (&tmp, tmp.blk);
    }

    if (err == SDC_OK) {
        memcpy(type, &tmp, sizeof(tmp));
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_type_set_block_mode(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t blk)
{
    sdc_error_t err = SDC_OK;
    uint32_t blk_flag;
    uint32_t blk_idx;
    uint32_t alg_idx;

    err = caam_map_sdc_wrap_unwrap_blk(blk, &blk_flag, &blk_idx);

    if (err == SDC_OK) {
        alg_idx = type->arch_alg_idx;

        /* invalid block mode */
        err = SDC_ALG_MODE_INVALID;

        /* check alg index - maybe someone has written directly to container */
        if (alg_idx < sizeof(caam_ee_cipher_descs) / sizeof(struct caam_ee_cipher_desc)) {
            /* check if block mode is supported */
            if ((caam_ee_cipher_descs[alg_idx].wrap_blk_mode_msk & (1<<blk_idx)) != 0) {
                type->blk = blk;
                type->arch_blk_flag = blk_flag;
                type->arch_blk_idx = blk_idx;
                err = SDC_OK;
            }
        }
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_type_get_algorithm(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t *alg)
{
    *alg = type->alg;

    return SDC_OK;

}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_type_get_block_mode(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t *blk)
{
    *blk = type->blk;

    return SDC_OK;
}

static size_t get_size_bit_mask_for_bytes_int (size_t bytes)
{
    size_t mask = 0;

    while (bytes > 0) {
        bytes--;
        mask = mask << 8;
        mask |= 0xFF;
    }

    return mask;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_key_desc_fill (
    const sdc_wrap_unwrap_type_t *type,
    sdc_key_desc_t *desc)
{
    uint32_t alg_idx;
    uint32_t blk_idx;

    blk_idx = type->arch_blk_idx;
    alg_idx = type->arch_alg_idx;

    /* check range of alg */
    if (alg_idx >= sizeof(caam_ee_cipher_descs) / sizeof(struct caam_ee_cipher_desc))
        return SDC_ALG_MODE_INVALID;

    /* check if block mode is supported */
    if ((caam_ee_cipher_descs[alg_idx].wrap_blk_mode_msk & (1<<blk_idx)) == 0)
        return SDC_ALG_MODE_INVALID;

    caam_keylen_bmsk_to_sdc(caam_ee_cipher_descs[alg_idx].key_len_msk,
                            &(desc->sup_key_lens), &(desc->dflt_key_len));
    desc->sup_key_fmt_protect   = SDC_KEY_FMT_SIMPLE_SYM;
    desc->sup_key_fmt_unprotect = SDC_KEY_FMT_SIMPLE_SYM;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_unwrap_desc_fill (
    sdc_session_t *session,
    const sdc_wrap_unwrap_type_t *type,
    sdc_wrap_unwrap_desc_t *desc,
    size_t total_data_len)
{
    const struct caam_ee_blk_desc *caam_ee_wrap_blk_desc;
    uint32_t blk_idx;
    uint32_t alg_idx;
    size_t len_bytes;

    (void)session;

    blk_idx = type->arch_blk_idx;
    alg_idx = type->arch_alg_idx;

    /* check range of alg */
    if (alg_idx >= sizeof(caam_ee_cipher_descs) / sizeof(struct caam_ee_cipher_desc))
        return SDC_ALG_MODE_INVALID;

    /* check if block mode is supported */
    if ((caam_ee_cipher_descs[alg_idx].wrap_blk_mode_msk & (1<<blk_idx)) == 0)
        return SDC_ALG_MODE_INVALID;

    caam_ee_wrap_blk_desc = &caam_ee_cipher_descs[alg_idx].wrap_blk_descs[blk_idx];

    desc->iv.max = caam_ee_wrap_blk_desc->iv_max;
    desc->iv.min = caam_ee_wrap_blk_desc->iv_min;
    desc->iv.mod = caam_ee_wrap_blk_desc->iv_mod;

    if ((type->alg == SDC_WRAP_ALG_AES) && (type->blk == SDC_WRAP_BLK_CCM)) {
        len_bytes = AES_CCM_TOTAL_HEADER_BYTES - desc->iv.max;
        while (total_data_len & (~get_size_bit_mask_for_bytes_int(len_bytes))) {
            len_bytes++;
            desc->iv.max--;
        }
    }
    desc->iv.dflt = desc->iv.max;

    desc->tag.max = caam_ee_wrap_blk_desc->tag_max;
    desc->tag.min = caam_ee_wrap_blk_desc->tag_min;
    desc->tag.mod = caam_ee_wrap_blk_desc->tag_mod;
    desc->tag.dflt = desc->tag.max;

    desc->data.block_len = caam_ee_cipher_descs[alg_idx].block_size;
    desc->data.max_chunk_len = caam_ee_wrap_blk_desc->max_block_len;
    desc->data.chunk_len_aligned = true;

    /* we don't require or support padding - AES-CCM is the only format */
    desc->data.padding = SDC_PADDING_NO;
    desc->supports_iuf = true;

    return sdc_intern_get_plain_cipher_spec(
               desc->data.padding,
               0,
               caam_ee_wrap_blk_desc->max_total_len,
               desc->data.block_len,
               &(desc->data.plain),
               &(desc->data.cipher));
}



/*
 * Only supported mode is AES-CCM
 *
 * Any length is supported
 * Only non final requests need to be aligned
 *
 * if op == WRAP
 *      tag_in_data is NULL
 *      tag_out_data is != NULL
 * else
 *      tag_in_data is != NULL
 *      tag_out_data is NULL
 */
static sdc_error_t caam_wrap_unwrap(sdc_session_t *session,
                                    caam_wrap_unwrap_op operation,
                                    const sdc_wrap_unwrap_type_t *type,
                                    const sdc_wrap_unwrap_desc_t *desc,
                                    const uint8_t *in_data,
                                    uint8_t *out_data,
                                    const size_t data_len,
                                    const uint8_t *tag_in_data, const size_t tag_in_len,
                                    const uint8_t *iv, const size_t iv_len,
                                    uint8_t *tag_out_data, const size_t tag_out_len)
{

    sdc_error_t err = SDC_OK;
    const uint8_t *in_data_ptr;
    uint8_t *out_data_ptr;
    size_t remaining_len;
    size_t out_len_remaining;
    size_t buf_idx;
    size_t buf_len;
    size_t align;
    struct caam_ee_wrap_unwrap_params iodata;
    int res;
    int res_errno;
    unsigned long int request;

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    in_data_ptr = in_data;
    out_data_ptr = out_data;
    remaining_len = data_len;
    out_len_remaining = data_len;
    buf_idx = 0;

    iodata.iv = (uint8_t*)iv;
    iodata.iv_len = (size_t)iv_len;

    iodata.total_len = (size_t)data_len;

    if (operation == WRAP) {
        iodata.tag = tag_out_data;
        iodata.tag_len = tag_out_len;
        iodata.desired_tag_len = tag_out_len;
        request = CAAM_WRAP_DATA;
    } else {
        iodata.tag = (uint8_t*)tag_in_data;
        iodata.tag_len = tag_in_len;
        iodata.desired_tag_len = tag_in_len;
        request = CAAM_UNWRAP_DATA;
    }

    iodata.flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_WRAP_INDATA_NORM_FLAG | CAAM_EE_WRAP_DEFAULT_FLAGS;

    while (((remaining_len > 0) || (buf_idx == 0)) && (err == SDC_OK)) {
        buf_len = remaining_len;

        if (buf_len > desc->data.max_chunk_len)
            buf_len = desc->data.max_chunk_len;

        remaining_len -= buf_len;

        iodata.flags &= ~CAAM_OP_MASK;

        if (remaining_len == 0) {
            /* final or init-final */
            if (buf_idx == 0) {
                iodata.flags |= CAAM_OP_SINGLE;
            } else {
                iodata.flags |= CAAM_OP_MULTI_FINALIZE;
            }
        } else {
            /* init or update */

            /* alignment */
            align = buf_len % desc->data.block_len;
            buf_len -= align;
            remaining_len += align;

            if (buf_idx == 0) {
                iodata.flags |= CAAM_OP_MULTI_INIT;
            } else {
                iodata.flags |= CAAM_OP_MULTI_UPDATE;
            }
        }

        iodata.in = (uint8_t*)in_data_ptr;
        iodata.out = out_data_ptr;
        iodata.in_len = buf_len;
        iodata.out_len = out_len_remaining;

        res = ioctl(session->arch.fd, request, &iodata);

        if (res == 0) {
            buf_idx++;
            in_data_ptr += buf_len;
            out_data_ptr += iodata.out_len;
            out_len_remaining -= iodata.out_len;
        } else {
            res_errno = errno;

            /* ignore and retry after EAGAIN */
            if (res_errno != EAGAIN) {
                err = error_from_ioctl_errno(res_errno, request, iodata.flags);
            }

            remaining_len += buf_len;
        }
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_get_out_len(sdc_session_t *session,
                                      const sdc_wrap_unwrap_type_t *type,
                                      const sdc_wrap_unwrap_desc_t *desc,
                                      const size_t in_len,
                                      size_t *out_len)
{
    (void)session;
    (void)type;

    return sdc_intern_get_cipher_len(
               desc->data.padding,
               in_len,
               desc->data.block_len,
               out_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap(sdc_session_t *session,
                          const sdc_wrap_unwrap_type_t *type,
                          const sdc_wrap_unwrap_desc_t *desc,
                          const uint8_t *in_data,
                          const size_t in_data_len,
                          uint8_t *out_data,
                          const size_t out_data_len,
                          const uint8_t *aad_data, const size_t aad_len,
                          const uint8_t *iv, const size_t iv_len,
                          uint8_t *tag_out_data, const size_t tag_out_len)
{

    (void)aad_data;

    if (aad_len != 0)
        return SDC_AAD_DATA_INVALID;

    /* we don't require or support padding - AES-CCM is the only format */
    if (in_data_len != out_data_len) {
        return SDC_INTERNAL_ERROR;
    }

    return caam_wrap_unwrap(session, WRAP,
                            type,
                            desc,
                            in_data,
                            out_data,
                            in_data_len,
                            NULL, 0,
                            iv, iv_len,
                            tag_out_data, tag_out_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_unwrap_get_max_out_len(sdc_session_t *session,
                                            const sdc_wrap_unwrap_type_t *type,
                                            const sdc_wrap_unwrap_desc_t *desc,
                                            const size_t in_len,
                                            size_t *out_max_len)
{
    (void)session;
    (void)type;

    return sdc_intern_get_max_plain_len(
               desc->data.padding,
               in_len,
               desc->data.block_len,
               out_max_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_unwrap(sdc_session_t *session,
                            const sdc_wrap_unwrap_type_t *type,
                            const sdc_wrap_unwrap_desc_t *desc,
                            const uint8_t *in_data,
                            const size_t in_data_len,
                            uint8_t *out_data,
                            size_t *out_data_len,
                            const uint8_t *aad_data, const size_t aad_len,
                            const uint8_t *tag_in_data, const size_t tag_in_len,
                            const uint8_t *iv, const size_t iv_len)
{

    (void)aad_data;
    if (aad_len != 0)
        return SDC_AAD_DATA_INVALID;

    /* we don't require or support padding - AES-CCM is the only format */
    if (*out_data_len != in_data_len)
        return SDC_INTERNAL_ERROR;

    return caam_wrap_unwrap(session, UNWRAP,
                            type,
                            desc,
                            in_data,
                            out_data,
                            in_data_len,
                            tag_in_data, tag_in_len,
                            iv, iv_len,
                            NULL, 0);
}

static sdc_error_t caam_arch_wrap_unwrap_init(sdc_session_t *session,
                                              const sdc_wrap_unwrap_desc_t *desc,
                                              const uint8_t *iv, size_t iv_len,
                                              int operation)
{
    sdc_error_t err = SDC_OK;

    if (desc->data.block_len > CAAM_EE_MAX_ENC_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_INTERNAL_ERROR;
    }

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    memcpy(session->arch.iv, (uint8_t *)iv, iv_len);
    session->arch.iv_len = (size_t) iv_len;

    if (operation == WRAP) {
        session->arch.request = CAAM_WRAP_DATA;
    } else {
        session->arch.request = CAAM_UNWRAP_DATA;
    }

    session->arch.buf_idx = 0;

    return err;
}

static sdc_error_t caam_wrap_unwrap_update(sdc_session_t *session,
                                           const sdc_wrap_unwrap_desc_t *desc,
                                           const sdc_wrap_unwrap_type_t *type,
                                           const uint8_t *in_data, const size_t in_data_len,
                                           uint8_t *out_data, size_t *out_data_len)
{
    sdc_error_t err = SDC_OK;
    size_t tag_len = 0;
    struct caam_ee_wrap_unwrap_params iodata;
    uint32_t iodata_flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_WRAP_INDATA_NORM_FLAG | CAAM_EE_WRAP_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res = -1;

    iodata.iv = (uint8_t*) session->arch.iv;
    iodata.iv_len = (size_t) session->arch.iv_len;
    /* total_len is actually needed only for init */
    iodata.total_len = session->iuf_wrap_unwrap.total_inlen;
    tag_len = session->iuf_wrap_unwrap.tag_len;

    /* output needs to be always greater or equal to input */
    if (in_data_len > *out_data_len)
        err = SDC_OUT_DATA_INVALID;

    /* data needs to be aligned */
    if ((in_data_len % desc->data.block_len) != 0)
        err = SDC_INTERNAL_ERROR;

    if (err == SDC_OK) {
        /* init or update */
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_MULTI_INIT;
        } else {
            iodata_flags |= CAAM_OP_MULTI_UPDATE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_wrap_unwrap_ioctl_operation(session, &iodata,
                                                           in_data, in_data_len,
                                                           NULL, tag_len,
                                                           NULL, tag_len,
                                                           out_data, *out_data_len,
                                                           iodata_flags);

            res_errno = errno;
            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, session->arch.request, iodata_flags);
                }
            }
        }
        if (err == SDC_OK) {
            session->arch.buf_idx++;
            *out_data_len = iodata.out_len;
        }
    }

    return err;
}

static sdc_error_t caam_wrap_unwrap_finalize(sdc_session_t *session,
                                             const sdc_wrap_unwrap_type_t *type,
                                             const sdc_wrap_unwrap_desc_t *desc,
                                             const uint8_t *in_data, const size_t in_data_len,
                                             uint8_t *tag_out_data, size_t tag_out_len,
                                             const uint8_t *tag_in_data, size_t tag_in_len,
                                             uint8_t *out_data, size_t *out_data_len)
{
    (void) desc;
    sdc_error_t err = SDC_OK;
    struct caam_ee_wrap_unwrap_params iodata;
    uint32_t iodata_flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_WRAP_INDATA_NORM_FLAG | CAAM_EE_WRAP_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;
    iodata.iv = (uint8_t*) session->arch.iv;
    iodata.iv_len = (size_t) session->arch.iv_len;
    /* total_len is actually needed only for init */
    iodata.total_len = session->iuf_wrap_unwrap.total_inlen;

    /* output needs to be always greater or equal to input */
    if (in_data_len > *out_data_len)
        err = SDC_OUT_DATA_INVALID;

    if (session->arch.buf_idx == 0) {
        iodata_flags |= CAAM_OP_SINGLE;
    } else {
        iodata_flags |= CAAM_OP_MULTI_FINALIZE;
    }

    /* retry in case of EAGAIN */
    while (err == SDC_OK) {
        res = caam_perform_wrap_unwrap_ioctl_operation(session, &iodata,
                                                       in_data, in_data_len,
                                                       tag_out_data,
                                                       tag_out_len, tag_in_data,
                                                       tag_in_len, out_data,
                                                       *out_data_len,
                                                       iodata_flags);
        res_errno = errno;
        if (res == 0) {
            break;
        } else {
            /* ignore and retry after EAGAIN */
            if (res_errno != EAGAIN) {
                err = error_from_ioctl_errno(res_errno, session->arch.request, iodata_flags);
            }
        }
    }

    if (err == SDC_OK) {
        session->arch.buf_idx++;
        *out_data_len = iodata.out_len;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_init(sdc_session_t *session,
                               const sdc_wrap_unwrap_type_t *type,
                               const sdc_wrap_unwrap_desc_t *desc,
                               const uint8_t *iv, size_t iv_len)
{
    (void)type;

    return caam_arch_wrap_unwrap_init(session, desc, iv, iv_len, WRAP);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_unwrap_init(sdc_session_t *session,
                                 const sdc_wrap_unwrap_type_t *type,
                                 const sdc_wrap_unwrap_desc_t *desc,
                                 const uint8_t *iv, size_t iv_len)
{
    (void)type;

    return caam_arch_wrap_unwrap_init(session, desc, iv, iv_len, UNWRAP);

}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_aad_update(sdc_session_t *session,
                                     const sdc_wrap_unwrap_type_t *type,
                                     const sdc_wrap_unwrap_desc_t *desc,
                                     const uint8_t *aad_data, const size_t aad_len)
{
    (void) session;
    (void) type;
    (void) desc;
    (void) aad_data;

    if (aad_len != 0)
        return SDC_AAD_DATA_INVALID;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_unwrap_aad_update(sdc_session_t *session,
                                       const sdc_wrap_unwrap_type_t *type,
                                       const sdc_wrap_unwrap_desc_t *desc,
                                       const uint8_t *aad_data, const size_t aad_len)
{
    (void) session;
    (void) type;
    (void) desc;
    (void) aad_data;

    if (aad_len != 0)
        return SDC_AAD_DATA_INVALID;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_update(sdc_session_t *session,
                                 const sdc_wrap_unwrap_type_t *type,
                                 const sdc_wrap_unwrap_desc_t *desc,
                                 const uint8_t *in_data, const size_t in_data_len,
                                 uint8_t *out_data, size_t *out_data_len)
{
    return caam_wrap_unwrap_update(session, desc, type,
                                   in_data,in_data_len,
                                   out_data, out_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_unwrap_update(sdc_session_t *session,
                                   const sdc_wrap_unwrap_type_t *type,
                                   const sdc_wrap_unwrap_desc_t *desc,
                                   const uint8_t *in_data, const size_t in_data_len,
                                   uint8_t *out_data, size_t *out_data_len)
{
    return caam_wrap_unwrap_update(session, desc, type,
                                   in_data,in_data_len,
                                   out_data, out_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_wrap_finalize(sdc_session_t *session,
                                   const sdc_wrap_unwrap_type_t *type,
                                   const sdc_wrap_unwrap_desc_t *desc,
                                   const uint8_t *in_data, const size_t in_data_len,
                                   uint8_t *tag_data, size_t tag_len,
                                   uint8_t *out_data, size_t *out_data_len)
{
    return caam_wrap_unwrap_finalize(session, type, desc,
                                     in_data, in_data_len,
                                     tag_data, tag_len,
                                     NULL, 0,
                                     out_data , out_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_unwrap_finalize(sdc_session_t *session,
                                     const sdc_wrap_unwrap_type_t *type,
                                     const sdc_wrap_unwrap_desc_t *desc,
                                     const uint8_t *in_data, const size_t in_data_len,
                                     const uint8_t *tag_data, size_t tag_len,
                                     uint8_t *out_data, size_t *out_data_len)
{
    return caam_wrap_unwrap_finalize(session, type, desc,
                                     in_data, in_data_len,
                                     NULL, 0,
                                     tag_data, tag_len,
                                     out_data, out_data_len);
}
